#!/usr/bin/python3

# ************************************************************
# Copyright © 2003-2025 Acronis International GmbH.
# This source code is distributed under MIT software license.
# ************************************************************

import ssl
import logging
import argparse
import asyncio
import server.routes as routes
from os.path import join, dirname, realpath
from aiohttp import web, ClientSession
from utils import sqlite_connect
from api_client import ApiClient
from constants import JWT_AUD, CLIENT_ID, CLIENT_SECRET, DC_URL

logging.basicConfig(format='[Service] %(asctime)s -- %(message)s', encoding='utf-8', level=logging.INFO)


@web.middleware
async def req_logger(request: web.Request, handler):
    logging.info(f'Headers {request.headers}')
    return await handler(request)


async def main(args):
    filename = join(dirname(realpath(__file__)), f'{args.db_name}.db')

    db = sqlite_connect(filename)

    app = web.Application(middlewares=[req_logger])
    app['db'] = db
    app['jwks'] = {'keys': []} # Will be filled on callback request

    routes.setup(app)

    async with ApiClient(DC_URL, CLIENT_ID, CLIENT_SECRET) as client:
        app['client'] = client
        runner = web.AppRunner(app)
        await runner.setup()
        if args.certfile and args.keyfile:
            ssl_ctx = ssl.create_default_context(ssl.Purpose.CLIENT_AUTH)
            ssl_ctx.load_cert_chain(args.certfile, args.keyfile)
            port = 443 if not args.port else args.port
            protocol = 'https'
            site = web.TCPSite(runner, port=port, ssl_context=ssl_ctx)
        else:
            port = 8080 if not args.port else args.port
            protocol = 'http'
            site = web.TCPSite(runner, port=port)
        await site.start()

        print(f'======== Running on {protocol}://0.0.0.0:{port} ========\n(Press CTRL+C to quit)')
        await asyncio.Event().wait()

    db.close()

if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('--certfile', help='Path to certificate\'s file.')
    parser.add_argument('--keyfile',  help='Path to certificate\'s private key.')
    parser.add_argument('--db-name',  help='Database name', default='vendor')
    parser.add_argument('--config-path',  help='Path to config file', default='connector.json')
    parser.add_argument('--port',  help='Web server\'s port')

    args = parser.parse_args()

    asyncio.run(main(args))
