# ************************************************************
# Copyright © 2003-2025 Acronis International GmbH.
# This source code is distributed under MIT software license.
# ************************************************************

import sqlite3
from typing import Optional
from datatypes import *
from aiohttp import web


def callback_topology_read_v2(conn: sqlite3.Connection, organization_id: str | None, request_id: str, response_id: str, context: CallbackContext, payload: Optional[dict]) -> web.Response:
    if not organization_id:
        return web.json_response(
            status=400,
            data=CallbackResponse(
                request_id=request_id,
                response_id=response_id,
                payload={'request_id': request_id, 'response_id': response_id, 'message': 'Cannot read non-existing partner.'}
            )
        )

    items: list[TopologyInfo] = [
        { 'id': row['id'], 'name': row['name'] }
        for row in conn.execute(
            'SELECT id, name FROM organizations WHERE parent_id = ? AND kind = ?',
            (organization_id, OrganizationKind.CUSTOMER)
        ).fetchall()
    ]
    return web.json_response(
        status=200,
        data=CallbackResponse(
            type='cti.a.p.acgw.response.v1.1~a.p.topology.read.ok.v1.0',
            request_id=request_id,
            response_id=response_id,
            payload=items
        )
    )

def callback_topology_read(conn: sqlite3.Connection, organization_id: str, request_id: str, response_id: str, context: CallbackContext, payload: Optional[dict]) -> web.Response:
    items: list[TopologyInfo] = [
        { 'id': row['id'], 'name': row['name'] }
        for row in conn.execute(
            '''WITH RECURSIVE organizations_tree AS (
                SELECT id, name, kind, parent_id AS direct_parent_id, parent_id FROM organizations

                UNION ALL

                SELECT p.id, p.name, p.kind, p.parent_id AS direct_parent_id, (SELECT parent_id FROM organizations WHERE id = p.parent_id) AS parent
                FROM organizations_tree AS p
                WHERE parent IS NOT NULL
            )
            SELECT id, name FROM organizations_tree WHERE parent_id = ? AND kind = ?''',
            (organization_id, OrganizationKind.CUSTOMER)
        ).fetchall()
    ]
    return web.json_response(
        status=200,
        data=CallbackResponse(
            type='cti.a.p.acgw.response.v1.0~a.p.topology.read.ok.v1.0',
            request_id=request_id,
            response_id=response_id,
            payload={ 'items': items }
        )
    )

# common callbacks
mapping = {
    'cti.a.p.acgw.callback.v1.0~a.p.topology.read.v1.0': callback_topology_read,
    'cti.a.p.acgw.callback.v2.0~a.p.topology.read.v1.0': callback_topology_read_v2,
}
