# ************************************************************
# Copyright © 2003-2025 Acronis International GmbH.
# This source code is distributed under MIT software license.
# ************************************************************

import sqlite3
from typing import Optional
from datatypes import *
from aiohttp import web


def callback_customer_mapping_get_state(conn: sqlite3.Connection, organization_id: str | None, request_id: str, response_id: str, context: CallbackContext, payload: Optional[dict]) -> web.Response:
    if not organization_id:
        pass

    # TODO: Settings are not set so we don't query them.
    items: list[OrganizationMappingPair] = [
        dict(row)
        for row in conn.execute(
            '''SELECT orgs.id AS vendor_tenant_id, map.acronis_tenant_id
            FROM organizations AS orgs
            JOIN organizations_mapping AS map ON orgs.id = map.organization_id
            WHERE (map.acronis_dc_url IS NULL OR map.acronis_dc_url = ?) AND orgs.parent_id = ? AND orgs.kind = ?''',
            (context['datacenter_url'], organization_id, OrganizationKind.CUSTOMER)
        ).fetchall()
    ]
    return web.json_response(
        status=200,
        data=CallbackResponse(
            type='cti.a.p.acgw.response.v1.1~a.p.customer.mapping.get_state.ok.v1.0',
            request_id=request_id,
            response_id=response_id,
            payload=items
        )
    )


def callback_customer_mapping_set_state(conn: sqlite3.Connection, organization_id: str | None, request_id: str, response_id: str, context: CallbackContext, payload: Optional[dict]) -> web.Response:
    if not organization_id:
        return web.json_response(
            status=400,
            data=CallbackResponse(
                request_id=request_id,
                response_id=response_id,
                payload={'request_id': request_id, 'response_id': response_id, 'message': 'Cannot map in non-existing partner.'}
            )
        )

    # NOTE: Here we assume a single level structure (Partner-Customer) and use the ID of the requesting partner tenant
    # to fetch direct customers.
    orgs: list[str] = [
        row['id'] for row in conn.execute(
            'SELECT id FROM organizations WHERE parent_id = ? AND kind = ?',
            (organization_id, OrganizationKind.CUSTOMER)
        ).fetchall()
    ]
    # There's nothing to map if there're no orgs available
    if not orgs:
        return web.json_response(status=400, data={'request_id': request_id, 'response_id': response_id, 'message': 'Nothing to map'})

    def conformant_map(payload: dict) -> bool:
        for item in payload.get('disabled', []):
            conn.execute('DELETE FROM organizations_mapping WHERE acronis_tenant_id = ?', (item['acronis_tenant_id'],))
        for item in payload.get('enabled', []):
            if item['vendor_tenant_id'] not in orgs:
                return False
            conn.execute('INSERT INTO organizations_mapping VALUES (?,?,?,?)', (item['vendor_tenant_id'], item['acronis_tenant_id'], context['datacenter_url'], None))
        return True

    res = conformant_map(payload)
    if not res:
        conn.rollback()
        return web.json_response(status=400, data={'request_id': request_id, 'response_id': response_id, 'message': 'Failed to write the mapping.'})

    conn.commit()
    return web.json_response(
        status=200,
        data=CallbackResponse(
            type='cti.a.p.acgw.response.v1.1~a.p.customer.mapping.set_state.ok.v1.0',
            request_id=request_id,
            response_id=response_id
        )
    )


# mapping v2 callbacks
mapping = {
    'cti.a.p.acgw.callback.v2.0~a.p.customer.mapping.get_state.v1.0': callback_customer_mapping_get_state,
    'cti.a.p.acgw.callback.v2.0~a.p.customer.mapping.set_state.v1.0': callback_customer_mapping_set_state
}
