# ************************************************************
# Copyright © 2003-2025 Acronis International GmbH.
# This source code is distributed under MIT software license.
# ************************************************************

import sqlite3
from typing import Optional
from uuid import uuid4
from datatypes import *
from aiohttp import web


def callback_mirroring_partner_get_state(conn: sqlite3.Connection, organization_id: str | None, request_id: str, response_id: str, context: CallbackContext, payload: dict) -> web.Response:
    if not organization_id:
        # If organization_id was not found, it is considered as disabled.
        return web.json_response(
            status=200,
            data=CallbackResponse(
                type=f'cti.a.p.acgw.response.v1.1~a.p.partner.mirroring.get_state.ok.v1.0',
                request_id=request_id,
                response_id=response_id,
                payload={
                    'state': 'DISABLED',
                }
            )
        )

    # TODO: Settings are not set so we don't query them.
    row = conn.execute('''SELECT o.external_id AS acronis_tenant_id, om.state
                       FROM organizations AS o
                       LEFT JOIN organizations_mirroring AS om ON o.external_id = om.id
                       WHERE o.id = ? AND o.kind = ?''', (organization_id, OrganizationKind.PARTNER)).fetchone()

    return web.json_response(
        status=200,
        data=CallbackResponse(
            type=f'cti.a.p.acgw.response.v1.1~a.p.partner.mirroring.get_state.ok.v1.0',
            request_id=request_id,
            response_id=response_id,
            payload={
                'state': row['state'],
                'acronis_tenant_id': row['acronis_tenant_id'],
                'vendor_tenant_id': organization_id,
            }
        )
    )

def callback_mirroring_partner_enable(conn: sqlite3.Connection, organization_id: str | None, request_id: str, response_id: str, context: CallbackContext, payload: dict) -> web.Response:
    if organization_id:
        # Return successful result to be fault-tolerant in case the partner was already mirrored.
        # Since organization is uniquely identified and secured by the token,
        # it will not be possible to take over the integration from another tenant or by crafting a malicious request.
        row = conn.execute('''SELECT o.external_id AS acronis_tenant_id, om.state
                       FROM organizations AS o
                       LEFT JOIN organizations_mirroring AS om ON o.external_id = om.id
                       WHERE o.id = ? AND o.kind = ?''', (organization_id, OrganizationKind.PARTNER)).fetchone()
        if row['state'] == 'DISABLED':
            conn.execute('''UPDATE organizations_mirroring
                    SET state = "ENABLED"
                    WHERE id = ?''', (row['acronis_tenant_id'],))
            conn.commit()
        return web.json_response(
            status=200,
            data=CallbackResponse(
                type=f'cti.a.p.acgw.response.v1.1~a.p.partner.mirroring.enable.ok.v1.0',
                request_id=request_id,
                response_id=response_id,
                payload={
                    'state': 'ENABLED',
                    'acronis_tenant_id': row['acronis_tenant_id'],
                    'vendor_tenant_id': organization_id,
                }
            )
        )

    vendor_tenant_id = str(uuid4())
    conn.execute('''INSERT INTO organizations_mirroring VALUES (?,?,?,?)''', (payload['acronis_tenant_id'], context['datacenter_url'], None, 'ENABLED'))
    conn.execute('''INSERT INTO organizations VALUES (?,?,?,?,?)''', (vendor_tenant_id, payload['acronis_tenant_id'], None, payload['acronis_tenant_name'], OrganizationKind.PARTNER))
    # TODO: Create a sample customer for the partner for demo mapping purposes.
    conn.execute('''INSERT INTO organizations VALUES (?,?,?,?,?)''', (str(uuid4()), None, vendor_tenant_id, 'Customer 1', OrganizationKind.CUSTOMER))
    conn.commit()

    return web.json_response(
        status=200,
        data=CallbackResponse(
            type=f'cti.a.p.acgw.response.v1.1~a.p.partner.mirroring.enable.ok.v1.0',
            request_id=request_id,
            response_id=response_id,
            payload={
                'state': 'ENABLED',
                'acronis_tenant_id': payload['acronis_tenant_id'],
                'vendor_tenant_id': vendor_tenant_id,
            }
        )
    )

def callback_mirroring_partner_reset(conn: sqlite3.Connection, organization_id: str | None, request_id: str, response_id: str, context: CallbackContext, payload: Optional[dict]) -> web.Response:
    # Return successful result to be fault-tolerant in case the partner was not mirrored.
    if organization_id:
        conn.execute('''UPDATE organizations_mirroring
                    SET state = "DISABLED"
                    WHERE id = (SELECT external_id FROM organizations WHERE id = ? AND kind = ?)''', (organization_id, OrganizationKind.PARTNER))
        conn.commit()

    return web.json_response(
        status=200,
        data=CallbackResponse(
            type=f'cti.a.p.acgw.response.v1.1~a.p.partner.mirroring.reset.ok.v1.0',
            request_id=request_id,
            response_id=response_id,
        )
    )

def callback_mirroring_customer_get_state(conn: sqlite3.Connection, organization_id: str | None, request_id: str, response_id: str, context: CallbackContext, payload: Optional[dict]) -> web.Response:
    if not organization_id:
        # If partner organization_id was not found, customer is considered as disabled.
        return web.json_response(
            status=200,
            data=CallbackResponse(
                type=f'cti.a.p.acgw.response.v1.1~a.p.customer.mirroring.get_state.ok.v1.0',
                request_id=request_id,
                response_id=response_id,
                payload=[]
            )
        )

    # TODO: Settings are not set so we don't query them.
    row = conn.execute('''SELECT o.id AS vendor_tenant_id, o.external_id AS acronis_tenant_id
                       FROM organizations AS o
                       LEFT JOIN organizations_mirroring AS om ON o.external_id = om.id
                       WHERE o.parent_id = ? AND o.kind = ? AND om.state = "ENABLED"''', (organization_id, OrganizationKind.CUSTOMER)).fetchall()

    return web.json_response(
        status=200,
        data=CallbackResponse(
            type=f'cti.a.p.acgw.response.v1.1~a.p.customer.mirroring.get_state.ok.v1.0',
            request_id=request_id,
            response_id=response_id,
            payload=[{'acronis_tenant_id': row['acronis_tenant_id'], 'vendor_tenant_id': row['vendor_tenant_id']} for row in row]
        )
    )

def callback_mirroring_customer_set_state(conn: sqlite3.Connection, organization_id: str | None, request_id: str, response_id: str, context: CallbackContext, payload: Optional[dict]) -> web.Response:
    # Return successful result to be fault-tolerant in case the customer was not mirrored.
    if not organization_id:
        return web.json_response(
            status=200,
            data=CallbackResponse(
                type=f'cti.a.p.acgw.response.v1.1~a.p.customer.mirroring.set_state.ok.v1.0',
                request_id=request_id,
                response_id=response_id,
            )
        )

    if 'enabled' in payload:
        enabled: list[dict] = payload['enabled']
        for item in enabled:
            # Insert customer mirroring record if it does not exist.
            row = conn.execute('SELECT COUNT(1) AS count FROM organizations_mirroring WHERE id = ?', (item['acronis_tenant_id'],)).fetchone()
            if not row['count']:
                conn.execute('INSERT INTO organizations_mirroring VALUES (?,?,?,?)', (item['acronis_tenant_id'], context['datacenter_url'], None, 'ENABLED'))
                conn.execute('INSERT INTO organizations VALUES (?,?,?,?,?)', (str(uuid4()), item['acronis_tenant_id'], organization_id, item['acronis_tenant_name'], OrganizationKind.CUSTOMER))
            # Update customer mirroring record if it exists.
            else:
                conn.execute('UPDATE organizations_mirroring SET state = "ENABLED" WHERE id = ?', (item['acronis_tenant_id'],))

    if 'disabled' in payload:
        disabled: list[str] = payload['disabled']
        for item in disabled:
            conn.execute('''UPDATE organizations_mirroring
                        SET state = "DISABLED"
                        WHERE id = (SELECT external_id FROM organizations WHERE id = ? AND kind = ?)''', (item['acronis_tenant_id'], OrganizationKind.CUSTOMER))
    conn.commit()

    return web.json_response(
        status=200,
        data=CallbackResponse(
            type=f'cti.a.p.acgw.response.v1.1~a.p.customer.mirroring.set_state.ok.v1.0',
            request_id=request_id,
            response_id=response_id,
        )
    )

# mirroring v2 callbacks
mapping = {
    'cti.a.p.acgw.callback.v2.0~a.p.partner.mirroring.get_state.v1.0': callback_mirroring_partner_get_state,
    'cti.a.p.acgw.callback.v2.0~a.p.partner.mirroring.enable.v1.0': callback_mirroring_partner_enable,
    'cti.a.p.acgw.callback.v2.0~a.p.partner.mirroring.reset.v1.0': callback_mirroring_partner_reset,
    'cti.a.p.acgw.callback.v2.0~a.p.customer.mirroring.get_state.v1.0': callback_mirroring_customer_get_state,
    'cti.a.p.acgw.callback.v2.0~a.p.customer.mirroring.set_state.v1.0': callback_mirroring_customer_set_state,
}
